/******************************************************************************
 *
 *   This file is part of SC1 - Studer Communication driver 1, by Sven Ruin
 *   and Anders Andersson.
 *
 *   Copyright (C) 2010-2019  TEROC AB.
 *
 *   SC1 is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   SC1 is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with SC1.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Note that SC1 depends on other software, with different licensing
 *   conditions.
 *
 *   We are thankful to Echelon/Adesto, Studer Innotec and others, who have
 *   provided helpful examples and support.
 *
 *   For more information, contact Sven Ruin, sven.ruin@teroc.se
 *
 *   See version number below in init section.
 *   
 *   Revision history:
 *   Ver 1.3 (2019-10-29) GNU General Public License, ++ operator, comments etc
 *   Ver 1.2 (2015-02-09) Changed to 1 req/s
 *   Ver 1.1 (2014-11-06) Corrected scaling of YD variables
 *
 *   SC1 is a FPM Driver intended to run on an i.LON SmartServer FT/PL
 *   connected to Studer XCOM-232i (or RCC-02-32) with a serial communication
 *   cable, with the following connections:
 *
 *   i.LON screw terminal        Studer side contact pin number
 *   23 (RXD)                    2
 *   24 (TXD)                    3
 *   25 (GND)                    5
 *
 ******************************************************************************/

#include "UFPTSC1.h"
// => section dependent includes
// <= section dependent includes
using namespace _0000000000000000_0_;

using namespace SmartServer::FPM_LIB_VERSION;
using namespace _9FFD030000000000_5__UFPTSC1_DRV;

// Sven's code - begin
#define SIO_HW_OPTS_SET		0x1005
// Sven's code - end

//-------------------------------------------------------------------
// => section datapoint variable declarations. DO NOT REMOVE THIS SECTION'S COMMENT!
DECLARE( _0000000000000000_0_::SNVT_volt, nvoUbat, OUTPUT_DP )            // 3000 f 2
DECLARE( _0000000000000000_0_::SNVT_temp, nvoTbat, OUTPUT_DP )            // 3001 f 2
DECLARE( _0000000000000000_0_::SNVT_amp, nvoIbat, OUTPUT_DP )             // 3005 f 1
DECLARE( _0000000000000000_0_::SNVT_count, nvoPhase, OUTPUT_DP )          // 3010 e 2
DECLARE( _0000000000000000_0_::SNVT_volt, nvoUin, OUTPUT_DP )             // 3011 f 2
DECLARE( _0000000000000000_0_::SNVT_amp, nvoIin, OUTPUT_DP )              // 3012 f 1
DECLARE( _0000000000000000_0_::SNVT_switch, nvoTransf, OUTPUT_DP )        // 3020 e 2
DECLARE( _0000000000000000_0_::SNVT_volt, nvoUout, OUTPUT_DP )            // 3021 f 2
DECLARE( _0000000000000000_0_::SNVT_amp, nvoIout, OUTPUT_DP )             // 3022 f 1
DECLARE( _0000000000000000_0_::SNVT_count, nvoMode, OUTPUT_DP )           // 3028 e 2
DECLARE( _0000000000000000_0_::SNVT_elec_whr_f, nvoEACinYD, OUTPUT_DP )   // 3080 f 3
DECLARE( _0000000000000000_0_::SNVT_elec_whr_f, nvoEACoutYD, OUTPUT_DP )  // 3082 f 3
DECLARE( _0000000000000000_0_::SNVT_switch, nvoCommOK, OUTPUT_DP )
// <= section datapoint variable declarations. DO NOT REMOVE THIS SECTION'S COMMENT!
//-------------------------------------------------------------------
// If variables are needed that accommodate values local to each fpm instance 
// these variables are declared here:
// DECLARE_FB_INSTANCE_LOCAL( int, _nInstanceNo )

// Sven's code - begin
DECLARE_FB_INSTANCE_LOCAL( int, rxBufCountOld)
DECLARE_FB_INSTANCE_LOCAL( int, fd)
DECLARE_FB_INSTANCE_LOCAL( int, tmr1)
DECLARE_FB_INSTANCE_LOCAL( int, rxTimeout)
DECLARE_FB_INSTANCE_LOCAL( int, rxRdy)
DECLARE_FB_INSTANCE_LOCAL( int, bytesExpected)
DECLARE_FB_INSTANCE_LOCAL( int, count)
// Sven's code - end

//-------------------------------------------------------------------
// ==> the one and only instance
//-------------------------------------------------------------------
static FPM::TStarter<CUFPTSC1> STARTER( FPM_MODULE_NAME );

//-------------------------------------------------------------------
// ==> extern "C" commands (typically not needed)
//-------------------------------------------------------------------
#ifdef __cplusplus
  extern "C" STATUS fnStartFPM_UFPTSC1() {
    return STARTER.StartModule();
  }
  
  //-------------------------------------------------------------------
  extern "C" STATUS fnStopFPM_UFPTSC1() { 
    return STARTER.StopModule();
  }
  
  //-------------------------------------------------------------------
  extern "C" void fnFPMLibVersion() { 
    STARTER.GetFPMLibVersion();
  }
#endif


//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// class: CUFPTSC1 
//+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
CUFPTSC1 *CUFPTSC1::m_pSingleton = NULL;

// -------------------------------------------------------------------
// Class: CUFPTSC1  [CStarter METHODs]
// -------------------------------------------------------------------
STATUS CUFPTSC1::StartModule(const char* pszModuleName)
{
  string      sInitString;
  E_ErrorCode eFaultCode = eFENone;
  CUFPTSC1::GetSingleton()->InitApp( sInitString, eFaultCode );
  if (eFENone == eFaultCode) {
    return OK;
  }
  printf ( "Error while starting FPM UFPTSC1: %s\n", sInitString.c_str() );

  return ERROR;
}

STATUS CUFPTSC1::StopModule(const char* pszModuleName)
{
  if (ERROR == CUFPTSC1::GetSingleton()->ShutdownBase()) {
    return ERROR;
  }
  CUFPTSC1::DestroySingleton();

  return OK;
}

// -------------------------------------------------------------------
// CUFPTSC1   [CONSTRUCTOR / DESTRUCTOR]
// -------------------------------------------------------------------
CUFPTSC1::CUFPTSC1()
  : CFPM_App( FPM_MODULE_NAME, CFPM_App::eDriver )

  // Sven's code - begin
  , m_oTimer1( this )
  // Sven's code - end
{
}

CUFPTSC1::~CUFPTSC1()
{
}

// This routine is called once, while starting up the module
void CUFPTSC1::Initialize()
{
  // Sven's code - begin
  int res;

  nvoEACinYD=0.0/0.0;   // NaN
  nvoEACoutYD=0.0/0.0;  // NaN

  printf("Init Studer Communication driver 1.3 \n");

  fd=rs232_open(38400);
  res=ioctl(fd, FIOSETOPTIONS, OPT_RAW);
  printf("SetRaw res=%i \n",res);
  res=ioctl(fd, SIO_HW_OPTS_SET, CLOCAL | CS8 | PARENB ); 	//sends e81, receives e81 (after restart)
  printf("SetOpts res=%i \n",res);

  m_oTimer1.Start(FPM_TF_REPEAT, 1000);
  // Sven's code - end
}

void CUFPTSC1::Shutdown()
{
  // Sven's code - begin
  StopAllTimers();
  rs232_close(fd);
  // Sven's code - end
}

void CUFPTSC1::Work()
{
}

void CUFPTSC1::OnTimer()
{
  // Sven's code - begin
		SNVT_switch sw_tmp;
		int bytesAvailable, bytesRead, i;
		int a, b, in;
		unsigned char rxbuf[200];
		unsigned char txbuf[26];
		unsigned char tmp[4];
		float *flt;
		static int commfault;
		static int fchlist[10][2] ={		//  {id,expectedlength}
			{3005,30},	// Ibat
			{3012,30},	// Iin
			{3020,28},	// Transf
			{3022,30},	// Iout
			};
		static int schlist[10][2] ={
			{3000,30},	// Ubat
			{3001,30},	// Tbat
			{3010,28},	// Phase
			{3011,30},	// Uin
			{3021,30},	// Uout
			{3028,28},	// Mode
			{3080,30},	// EACinYD
			{3082,30},	// EACoutYD
			};
		static int fptr=0, sptr=0, id;

		if (m_oTimer1.Expired()){
			ioctl(fd, FIONREAD, (int) &bytesAvailable);

			if(bytesAvailable>0 ){
				if(bytesAvailable>bytesExpected) RxbufClean(fd);
				if(bytesAvailable==bytesExpected){
					count=count+1;
					bytesRead=rs232_read(fd,rxbuf,bytesAvailable);
					rxRdy=1;
					rxTimeout=0;

					a=255;      //calculate header checksum
					b=0;
					for(i=1;i<12;i++){
						a=a+rxbuf[i];
						b=b+a;
					}
					if(rxbuf[12]==(Byte)a && rxbuf[13]==(Byte)b){
	//					printf("Chk 1 OK \n");
						a=255;  //calculate data checksum
						b=0;
						for(i=14;i<14+rxbuf[10];i++){
							a=a+rxbuf[i];
							b=b+a;
						}
	 					if(rxbuf[14+rxbuf[10]]==(Byte)a && rxbuf[15+rxbuf[10]]==(Byte)b){

							if( bytesExpected==28 ) in=rxbuf[25]*256+rxbuf[24];
							if( bytesExpected==30 ){   //change order of bytes for reading of float
								tmp[0]=rxbuf[27];
								tmp[1]=rxbuf[26];
								tmp[2]=rxbuf[25];
								tmp[3]=rxbuf[24];
								flt=(float*) &tmp[0];
							}

							commfault=0;
							sw_tmp.state=1;
							sw_tmp.value=200;
							nvoCommOK=sw_tmp;
							switch (id){
								case 3000: nvoUbat=(int)(*flt*10);break;
								case 3001: nvoTbat=(int)(*flt*10+2740);break;
								case 3005: nvoIbat=(int)(*flt*10);break;
								case 3010: nvoPhase=in;break;
								case 3011: nvoUin=(int)(*flt*10);break;
								case 3012: nvoIin=(int)(*flt*10);break;
								case 3020:
									sw_tmp.state=in;
									sw_tmp.value=in*200;
									nvoTransf=sw_tmp;
									break;
								case 3021: nvoUout=(int)(*flt*10);break;
								case 3022: nvoIout=(int)(*flt*10);break;
								case 3028: nvoMode=in;break;
								case 3080: nvoEACinYD=*flt*1000;break;
								case 3082: nvoEACoutYD=*flt*1000;break;
							}

							if(id==3000){
	//							printf("Value %i (nvoUbat) %f \n",id,*flt);
	//							printf("Value %u %u %u %u \n",rxbuf[24],rxbuf[25],rxbuf[26],rxbuf[27]);
							}
						}
					}
					else{
						++commfault;
						printf("Checksum error \n");
					}
				}
			}
			rxBufCountOld=bytesAvailable;

			if(rxRdy==1){
				RxbufClean(fd);

				// ask first all fast channels, than one slow
				if( fchlist[fptr][0]==0 ){
					fptr=0;
					if( schlist[sptr][0]==0 ) sptr=0;
					id=schlist[sptr][0];
					bytesExpected=schlist[sptr][1];
					++sptr;
				}
				else{
					id=fchlist[fptr][0];
					bytesExpected=fchlist[fptr][1];
					++fptr;
				}

				txbuf[0]=0xAA;	//start_byte
				txbuf[1]=0x00;	//frame_flags
				txbuf[2]=0x01;	//src_addr
				txbuf[3]=0x00;
				txbuf[4]=0x00;
				txbuf[5]=0x00;
				txbuf[6]=0x65;	//dest_addr
				txbuf[7]=0x00;
				txbuf[8]=0x00;
				txbuf[9]=0x00;
				txbuf[10]=0x0A;	//data_length
				txbuf[11]=0x00;
				a=255;
				b=0;
				for(i=1;i<12;i++){
					a=a+txbuf[i];
					a=a & 0xFF;
					b=b+a;
					b=b & 0xFF;
				}
				txbuf[12]=(Byte)a;
				txbuf[13]=(Byte)b;

				txbuf[14]=0x00;	//flags
				txbuf[15]=0x01;	//read_property
				txbuf[16]=0x01;	//system_state
				txbuf[17]=0x00;
				txbuf[18]=id & 255;	//object_id
				txbuf[19]=id / 256;
				txbuf[20]=0x00;
				txbuf[21]=0x00;
				txbuf[22]=0x01;	//property_id
				txbuf[23]=0x00;
				a=255;
				b=0;
				for(i=14;i<24;i++){
					a=a+txbuf[i];
					b=b+a;
				}
				txbuf[24]=a;
				txbuf[25]=b;

				rs232_write(fd, txbuf, 26);
				rxRdy=0;
				rxTimeout=0;
			}

			if(rxTimeout<100)
			  rxTimeout=rxTimeout+1;
			if(rxTimeout==20){
			  rxRdy=1;
			  ++commfault;
			}

			if(commfault>5){
				commfault=30;
				printf("Comm.error \n");
				sw_tmp.state=0;
				sw_tmp.value=0;
				nvoCommOK=sw_tmp;
	// Default values at comm.error:
				nvoUbat=-32768;
				nvoTbat=0;
				nvoIbat=-32768;
				nvoPhase=0;
				nvoUin=-32768;
				nvoIin=-32768;
				sw_tmp.state=0;
				sw_tmp.value=0;
				nvoTransf=sw_tmp;
				nvoUout=-32768;
				nvoIout=-32768;
				nvoMode=0;
				nvoEACinYD=0.0/0.0;   // NaN
				nvoEACoutYD=0.0/0.0;  // NaN

			}
		}
  // Sven's code - end
}

// Sven's code - begin

int CUFPTSC1::BytesReadyForRead( HANDLE handle ) const
{
	int bytesAvailable;

    ioctl( handle, FIONREAD, (int) &bytesAvailable );
    return bytesAvailable;
}

void CUFPTSC1::RxbufClean( HANDLE handle ) const
{
	Byte rxbuf[200];
	int bytesAvailable, bytesRead;
	ioctl(fd, FIONREAD, (int) &bytesAvailable);
	while(bytesAvailable>0){
		if(bytesAvailable>100){
			bytesRead=rs232_read(fd,rxbuf,100);
			printf("Clearing rxbuf(1) %i \n",bytesRead);
		}
		else{
			bytesRead=rs232_read(fd,rxbuf,bytesAvailable);
			printf("Clearing rxbuf(2) %i \n",bytesRead);
			printf("Rx0 = %c \n",rxbuf[0]);
			printf("Rx1 = %c \n",rxbuf[1]);
			printf("Rx2 = %c \n",rxbuf[2]);

		}
		ioctl(fd, FIONREAD, (int) &bytesAvailable);
	}
}
// Sven's code - end
